#!/bin/bash
# ==========================================
# Aris Mac Setup - Run this BEFORE opening Aris
# ==========================================
# This script installs PostgreSQL + pgvector and creates
# the Aris database user. Just double-click or run in Terminal.
# ==========================================

clear
echo "=========================================="
echo "  🤖 Aris - Mac Setup"
echo "=========================================="
echo ""
echo "This will install everything Aris needs."
echo "You may be asked for your Mac password."
echo ""

# -----------------------------------------
# Step 1: Install Homebrew if missing
# -----------------------------------------
if ! command -v brew &> /dev/null; then
    echo "[1/4] Installing Homebrew (Mac package manager)..."
    /bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)"
    
    # Add to PATH for Apple Silicon
    if [ -f /opt/homebrew/bin/brew ]; then
        eval "$(/opt/homebrew/bin/brew shellenv)"
        echo 'eval "$(/opt/homebrew/bin/brew shellenv)"' >> ~/.zprofile
    fi
    echo "✅ Homebrew installed"
else
    echo "[1/4] Homebrew already installed ✅"
fi

# Make sure brew is in PATH for this session
if [ -f /opt/homebrew/bin/brew ]; then
    eval "$(/opt/homebrew/bin/brew shellenv)"
fi

# -----------------------------------------
# Step 2: Install PostgreSQL 17
# -----------------------------------------
if brew list postgresql@17 &> /dev/null; then
    echo "[2/4] PostgreSQL 17 already installed ✅"
else
    echo "[2/4] Installing PostgreSQL 17..."
    brew install postgresql@17
    echo "✅ PostgreSQL 17 installed"
fi

# Add PostgreSQL to PATH for this session AND permanently
export PATH="/opt/homebrew/opt/postgresql@17/bin:$PATH"
if ! grep -q "postgresql@17" ~/.zprofile 2>/dev/null; then
    echo 'export PATH="/opt/homebrew/opt/postgresql@17/bin:$PATH"' >> ~/.zprofile
fi

# Start PostgreSQL and set it to auto-start on boot
echo "     Starting PostgreSQL..."
brew services start postgresql@17
sleep 3
echo "     ✅ PostgreSQL running (auto-starts on reboot)"

# -----------------------------------------
# Step 3: Install pgvector
# -----------------------------------------
if brew list pgvector &> /dev/null; then
    echo "[3/4] pgvector already installed ✅"
else
    echo "[3/4] Installing pgvector..."
    brew install pgvector
    echo "✅ pgvector installed"
fi

# -----------------------------------------
# Step 4: Create Aris database user
# -----------------------------------------
echo "[4/4] Setting up Aris database..."

# Wait for PostgreSQL to be ready
for i in {1..10}; do
    if pg_isready -q 2>/dev/null; then
        break
    fi
    sleep 1
done

# Check if user already exists
if psql postgres -tAc "SELECT 1 FROM pg_roles WHERE rolname='aris'" 2>/dev/null | grep -q 1; then
    echo "     Aris user already exists ✅"
else
    psql postgres -c "CREATE USER aris WITH SUPERUSER PASSWORD 'aris123';" 2>/dev/null
    if [ $? -eq 0 ]; then
        echo "     ✅ Aris database user created"
    else
        echo "     ⚠️  Could not create user. Trying with default postgres user..."
        psql -U postgres -c "CREATE USER aris WITH SUPERUSER PASSWORD 'aris123';" 2>/dev/null
        if [ $? -eq 0 ]; then
            echo "     ✅ Aris database user created"
        else
            echo "     ❌ Failed to create user. You may need to do this manually."
            echo "        Open Terminal and run:"
            echo "        psql postgres -c \"CREATE USER aris WITH SUPERUSER PASSWORD 'aris123';\""
        fi
    fi
fi

# Create the database if it doesn't exist
if psql -U aris -lqt 2>/dev/null | cut -d \| -f 1 | grep -qw aris_local; then
    echo "     aris_local database already exists ✅"
else
    createdb -U aris aris_local 2>/dev/null
    if [ $? -eq 0 ]; then
        echo "     ✅ aris_local database created"
    else
        echo "     (Aris will create the database automatically on first launch)"
    fi
fi

# Enable pgvector extension
psql -U aris -d aris_local -c "CREATE EXTENSION IF NOT EXISTS vector;" 2>/dev/null

# -----------------------------------------
# Done!
# -----------------------------------------
echo ""
echo "=========================================="
echo "  ✅ SETUP COMPLETE!"
echo "=========================================="
echo ""
echo "  PostgreSQL is running and will auto-start."
echo "  You can now open Aris from Applications."
echo ""
echo "  If you ever need to check PostgreSQL status:"
echo "    brew services list"
echo ""
echo "=========================================="
echo ""
read -p "Press Enter to close..."
